//=============================================================================
// vga_top.v
// 
// This module is the top-level template module for hardware to control a
// VGA video interface.
// 
// 2022/04/04  Written [Ziyuan Dong]
//=============================================================================
module vga_top (
   //--- Clock Input   
   CLOCK_50,                // 50 MHz
	
   //--- Push Button  
   KEY,                     // Pushbutton[3:0]
	
   //--- LEDRs         
   LEDR,                    // LEDR[9:0]
	
   //--- Switch  
   SW,                      // Switch[9:0]
	
   //--- 7-segment hexadecimal displays 
	HEX0,                    // HEX0[7:0]
	HEX1,
	HEX2,
	HEX3,
	HEX4,
	HEX5,
	
   //--- VGA  
   VGA_CLK,                 // VGA clock
   VGA_HS,                  // VGA H_SYNC
   VGA_VS,                  // VGA V_SYNC
   VGA_BLANK_N,             // VGA BLANK
   VGA_SYNC_N,              // VGA SYNC
   VGA_R,                   // VGA Red[9:0]
   VGA_G,                   // VGA Green[9:0]
   VGA_B                    // VGA Blue[9:0]
);

//--- 50 MHz clock from DE1-SoC board
input          CLOCK_50;

//--- 10 Switches
input    [9:0] SW;

//--- 4 Push buttons
input    [3:0] KEY;

//--- 10 LEDs
output   [9:0] LEDR;

//--- 6 7-segment hexadecimal displays
output   [7:0] HEX0;        // seven segment digit 0
output   [7:0] HEX1;        // seven segment digit 1
output   [7:0] HEX2;        // seven segment digit 2
output   [7:0] HEX3;        // seven segment digit 3
output   [7:0] HEX4;        // seven segment digit 4
output   [7:0] HEX5;        // seven segment digit 5


//--- VGA  
output         VGA_CLK;      // VGA clock
output         VGA_HS;       // VGA H_SYNC
output         VGA_VS;       // VGA V_SYNC
output         VGA_BLANK_N;  // VGA BLANK
output         VGA_SYNC_N;   // VGA SYNC
output   [7:0] VGA_R;        // VGA Red[9:0]
output   [7:0] VGA_G;        // VGA Green[9:0]
output   [7:0] VGA_B;        // VGA Blue[9:0]

//=============================================================================
// reg/wire declarations
//=============================================================================
wire   [12:0] cam_col;
wire   [12:0] cam_row;

wire          clk_vga;
wire          VGA_CTRL_CLK;
wire          orequest;
wire          reset_n;

wire   [7:0]  VGA_R;
wire   [7:0]  VGA_G;
wire   [7:0]  VGA_B; 

reg    [7:0]  red_c;
reg    [7:0]  blue_c;
reg    [7:0]  green_c;

reg    [7:0]  red;
reg    [7:0]  green;
reg    [7:0]  blue;

reg    [12:0] o_hcont;
reg    [12:0] o_vcont;

//=============================================================================
// Main body
//=============================================================================
assign reset_n  = KEY[0];
assign VGA_CLK	 = clk_vga;

//--- Turn on LED[0] when SW[9] up
assign LEDR[0] = SW[9];

//--- Turn on HEX0[0] when KEY[3] pressed
assign  HEX0[0] = KEY[3];

pll p1(
   .refclk(CLOCK_50),   //  refclk.clk
	.rst(~reset_n),      //   reset.reset
	.outclk_0(clk_vga)   // outclk0.clk
);

// Main logic to set colors of pixels
always @(*) begin
   // default background color
   red_c   = 8'hb3;
   green_c = 8'he5;
   blue_c  = 8'hff;

   // 3 lines
   if (cam_row >= 13'd0389) begin
      red_c   = 8'hff;
      green_c = 8'h00;
      blue_c  = 8'h00;
   end
   if (cam_row >= 13'd0419) begin
      red_c   = 8'h00;
      green_c = 8'hff;
      blue_c  = 8'h00;
   end
   if (cam_row >= 13'd0449) begin
      red_c   = 8'h00;
      green_c = 8'h00;
      blue_c  = 8'hff;
   end

   // one rectangle
   if ((cam_row >= 13'd0200) && (cam_row <= 13'd0465) && (cam_col >= 13'd0200) && (cam_col <= 13'd0275)) begin
      red_c   = 8'hff;
      green_c = 8'h19;
      blue_c  = 8'hf0;
   end
end

// Flip-flop registers
always @(posedge CLOCK_50) begin
   if (clk_vga == 1'b0) begin
      red   <= #1 red_c;
      green <= #1 green_c;
      blue  <= #1 blue_c;
   end

end

//	H_Sync Generator, Ref. 25 MHz Clock
always@(posedge clk_vga or negedge reset_n)
begin
	if(!reset_n)
	begin
		o_hcont   <=	1'b0;
	end
	else
	begin
		//	H_Sync Counter
		if( o_hcont < 13'd800 )
			o_hcont   <=	o_hcont+1;
		else
			o_hcont   <=	1'b0;
	end
end

//	V_Sync Generator, Ref. H_Sync
always@(posedge clk_vga or negedge reset_n)
begin
	if(!reset_n)
	begin
		o_vcont   <=	1'b0;
	end
	else
	begin
		//	When H_Sync Re-start
		if(o_hcont==1'b0)
		begin
			//	V_Sync Counter
			if( o_vcont < 13'd525 )
				o_vcont	<=	o_vcont+1'b1;
			else
				o_vcont	<=	1'b0;
		end
	end
end

//VGA_controller
VGA_controller vga_control ( 
   // Host Side
   .orequest(orequest),
   .ired(red),
   .igreen(green),
   .iblue(blue),
   .o_vcont(o_vcont),
	.o_hcont(o_hcont),
   // VGA Side
   .ored(VGA_R),
   .ogreen(VGA_G),
   .oblue(VGA_B),
   .ovga_h_sync(VGA_HS),
   .ovga_v_sync(VGA_VS),
   .ovga_sync(VGA_SYNC_N),
   .ovga_blank(VGA_BLANK_N),
   .ocol(cam_col),
   .orow(cam_row),

   // Control signals
   .clk_vga(clk_vga),
   .reset_n(reset_n)
);
endmodule
