//=============================================================================
// vga_top.v
// 
// This module is the top-level template module for hardware to control a
// VGA video interface.
// 
// 2022/02/10  Written [Ziyuan Dong]
// 2022/05/03  Added HEX ports; Added a small LED, HEX, SW, KEY logic [Ziyuan Dong]
//=============================================================================
module vga_top (
   //--- Clock Input   
   CLOCK_50,                // 50 MHz
	
   //--- Push Button  
   KEY,                     // Pushbutton[3:0]
	
   //--- LEDRs         
   LEDR,                    // LEDR[9:0]
	
   //--- Switch  
   SW,                      // Switch[9:0]
	
   //--- 7-segment hexadecimal displays 
	HEX0,                    // HEX0[7:0]
	HEX1,
	HEX2,
	HEX3,
	HEX4,
	HEX5,
	
   //--- VGA  
   VGA_CLK,                 // VGA clock
   VGA_HS,                  // VGA H_SYNC
   VGA_VS,                  // VGA V_SYNC
   VGA_BLANK_N,             // VGA BLANK
   VGA_SYNC_N,              // VGA SYNC
   VGA_R,                   // VGA Red[9:0]
   VGA_G,                   // VGA Green[9:0]
   VGA_B                    // VGA Blue[9:0]
);

//--- 50 MHz clock from DE1-SoC board
input          CLOCK_50;

//--- 10 Switches
input    [9:0] SW;

//--- 4 Push buttons
input    [3:0] KEY;

//--- 10 LEDs
output   [9:0] LEDR;

//--- 6 7-segment hexadecimal displays
output   [7:0] HEX0;        // seven segment digit 0
output   [7:0] HEX1;        // seven segment digit 1
output   [7:0] HEX2;        // seven segment digit 2
output   [7:0] HEX3;        // seven segment digit 3
output   [7:0] HEX4;        // seven segment digit 4
output   [7:0] HEX5;        // seven segment digit 5


//--- VGA  
output         VGA_CLK;      // VGA clock
output         VGA_HS;       // VGA H_SYNC
output         VGA_VS;       // VGA V_SYNC
output         VGA_BLANK_N;  // VGA BLANK
output         VGA_SYNC_N;   // VGA SYNC
output   [7:0] VGA_R;        // VGA Red[9:0]
output   [7:0] VGA_G;        // VGA Green[9:0]
output   [7:0] VGA_B;        // VGA Blue[9:0]

//=============================================================================
// reg/wire declarations
//=============================================================================
wire [12:0] col;
wire [12:0] row;

wire       clk_vga;
wire       VGA_CTRL_CLK;
wire       orequest;
wire       reset_n;

wire [7:0] VGA_R;
wire [7:0] VGA_G;
wire [7:0] VGA_B; 

reg  [7:0] red_c;
reg  [7:0] blue_c;
reg  [7:0] green_c;

reg  [7:0] red;
reg  [7:0] green;
reg  [7:0] blue;

reg  [1:0] rClk, rClk_c;

//=============================================================================
// Main body
//=============================================================================
assign reset_n  = KEY[0];
assign VGA_CLK	= ~rClk[0];

//--- Turn on LED[0] when SW[9] up
assign LEDR[0] = SW[9];

//--- Turn on HEX0[0] when KEY[3] pressed
assign  HEX0[0] = KEY[3];


// rClk 2-bit counter
always @(*) begin
   rClk_c = rClk + 2'b01;

   if (reset_n == 1'b0) begin
      rClk_c = 2'b00;
   end
end

// Main logic to set colors of pixels
always @(*) begin
   // default background color
   red_c   = 8'hb3;
   green_c = 8'he5;
   blue_c  = 8'hff;

   // 3 lines
   if (row >= 13'd0389) begin
      red_c   = 8'hff;
      green_c = 8'h00;
      blue_c  = 8'h00;
   end
   if (row >= 13'd0419) begin
      red_c   = 8'h00;
      green_c = 8'hff;
      blue_c  = 8'h00;
   end
   if (row >= 13'd0449) begin
      red_c   = 8'h00;
      green_c = 8'h00;
      blue_c  = 8'hff;
   end

   // one rectangle
   if ((row >= 13'd0200) && (row <= 13'd0465) && (col >= 13'd0200) && (col <= 13'd0275)) begin
      red_c   = 8'hff;
      green_c = 8'h19;
      blue_c  = 8'hf0;
   end
end

// Flip-flop registers
always @(posedge CLOCK_50) begin
   if (rClk[0] == 1'b0) begin
      red   <= #1 red_c;
      green <= #1 green_c;
      blue  <= #1 blue_c;
   end

   rClk <= #1 rClk_c;
end

// Controller sub-module
assign clk_vga = rClk[0];

VGA_controller vga_control ( 
   // Host Side
   .orequest(orequest),
   .ired({2'b00,red[7:0]}),
   .igreen({2'b00,green[7:0]}),
   .iblue({2'b00,blue[7:0]}),

   // VGA Side
   .ored(VGA_R),
   .ogreen(VGA_G),
   .oblue(VGA_B),
   .ovga_h_sync(VGA_HS),
   .ovga_v_sync(VGA_VS),
   .ovga_sync(VGA_SYNC_N),
   .ovga_blank(VGA_BLANK_N),
   .col(col),
   .row(row),

   // Control signals
   .clk_vga(clk_vga),
   .reset_n(reset_n),
);
endmodule
